#!/usr/bin/env php
<?php
// cron/fetch_data.php - Fetch data from Binance and store in database

require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/BinanceAPI.php';
require_once __DIR__ . '/../includes/Database.php';
require_once __DIR__ . '/../includes/AlertChecker.php';
require_once __DIR__ . '/../includes/functions.php';

$api = new BinanceAPI();
$db = Database::getInstance();

echo "[" . date('Y-m-d H:i:s') . "] Starting data fetch...\n";

try {
    // Update futures symbols
    echo "Updating futures symbols...\n";
    $exchangeInfo = $api->getExchangeInfo();
    $symbols = [];
    
    foreach ($exchangeInfo['symbols'] as $symbol) {
        if ($symbol['status'] === 'TRADING' && 
            strpos($symbol['symbol'], 'USDT') !== false && 
            $symbol['contractType'] === 'PERPETUAL' &&
            strpos($symbol['symbol'], '_') === false) {
            
            $symbols[] = $symbol['symbol'];
        }
    }
    
    // Clear and insert symbols
    $pdo = $db->getConnection();
    $pdo->exec("TRUNCATE TABLE futures_symbols");
    
    $stmt = $pdo->prepare("INSERT INTO futures_symbols (symbol) VALUES (?)");
    foreach ($symbols as $symbol) {
        $stmt->execute([$symbol]);
    }
    
    echo "Updated " . count($symbols) . " symbols\n";
    
    // Get tickers
    echo "Fetching 24hr tickers...\n";
    $tickers = $api->get24hrTicker();
    
    // Get top 100 by volume
    $topSymbols = [];
    foreach ($tickers as $ticker) {
        if (in_array($ticker['symbol'], $symbols)) {
            $topSymbols[] = [
                'symbol' => $ticker['symbol'],
                'volume' => floatval($ticker['quoteVolume'])
            ];
        }
    }
    
    usort($topSymbols, function($a, $b) {
        return $b['volume'] <=> $a['volume'];
    });
    
    $topSymbols = array_slice($topSymbols, 0, 100);
    
    // Fetch RSI data for top symbols
    echo "Fetching RSI data...\n";
    foreach ($topSymbols as $symbolData) {
        $symbol = $symbolData['symbol'];
        
        try {
            $candles15m = $api->getKlines($symbol, '15m', 100);
            $candles1d = $api->getKlines($symbol, '1d', 50);
            
            if ($candles15m && $candles1d) {
                $closes15m = array_column($candles15m, 4);
                $closes1d = array_column($candles1d, 4);
                
                $rsi15m = calculateWilderRSI($closes15m, 7);
                $rsi1d = calculateWilderRSI($closes1d, 7);
                
                // Find ticker data
                $ticker = array_filter($tickers, function($t) use ($symbol) {
                    return $t['symbol'] === $symbol;
                });
                $ticker = reset($ticker);
                
                if ($ticker) {
                    $data = [
                        'symbol' => $symbol,
                        'rsi15m' => $rsi15m,
                        'rsi1d' => $rsi1d,
                        'price' => floatval($ticker['lastPrice']),
                        'volume' => floatval($ticker['quoteVolume']),
                        'change' => floatval($ticker['priceChangePercent'])
                    ];
                    
                    $db->saveRSIData($symbol, $data);
                }
            }
            
            echo ".";
        } catch (Exception $e) {
            echo "Error for {$symbol}: " . $e->getMessage() . "\n";
        }
        
        usleep(200000); // 200ms delay to avoid rate limits
    }
    
    echo "\nData fetch completed successfully!\n";
    
} catch (Exception $e) {
    echo "ERROR: " . $e->getMessage() . "\n";
    exit(1);
}