<?php
// includes/AlertChecker.php

require_once __DIR__ . '/BinanceAPI.php';
require_once __DIR__ . '/Database.php';
require_once __DIR__ . '/NotificationService.php';

class AlertChecker {
    private $api;
    private $db;
    private $notifier;
    private $userId;
    private $settings;
    
    public function __construct($userId = null) {
        $this->api = new BinanceAPI();
        $this->db = Database::getInstance();
        $this->notifier = new NotificationService();
        $this->userId = $userId;
        
        if ($userId) {
            $this->settings = $this->db->getUserSettings($userId);
        }
    }
    
    /**
     * Check all alerts for a user
     */
    public function checkAllAlerts($userId = null) {
        if ($userId) {
            $this->userId = $userId;
            $this->settings = $this->db->getUserSettings($userId);
        }
        
        if (!$this->settings || !$this->settings['enabled']) {
            return;
        }
        
        // Get symbols with volume
        $symbols = $this->getSymbolsWithVolume();
        
        // Get tracked coins for this user
        $trackedCoins = $this->db->getTrackedCoins($this->userId);
        $trackedSymbols = array_keys($trackedCoins);
        
        // Check each symbol
        foreach ($symbols as $symbolData) {
            $symbol = $symbolData['symbol'];
            
            // Skip if not tracked and filtering is enabled with selected coins
            if ($this->settings['alert_filtering_enabled'] && 
                !empty($this->settings['selectedCoins']) && 
                !in_array($symbol, $this->settings['selectedCoins'])) {
                continue;
            }
            
            // Check all alert types
            $this->checkSymbolAlerts($symbol, $symbolData, $trackedSymbols);
        }
    }
    
    /**
     * Check alerts for a specific symbol
     */
    private function checkSymbolAlerts($symbol, $symbolData, $trackedSymbols = []) {
        // Get additional data
        $candles15m = $this->api->getKlines($symbol, '15m', 100);
        $candles1h = $this->api->getKlines($symbol, '1h', 48);
        $candles4h = $this->api->getKlines($symbol, '4h', 30);
        $candles1d = $this->api->getKlines($symbol, '1d', 50);
        
        // Calculate RSI
        $rsiData = $this->calculateRSI($symbol, $candles15m, $candles1d);
        
        // Calculate support/resistance
        $srLevels = $this->calculateSupportResistance($candles15m);
        
        // Analyze trend
        $trend = $this->analyzeTrend($candles1h);
        
        $fullData = array_merge($symbolData, $rsiData, $srLevels, [
            'trend' => $trend,
            'candles15m' => $candles15m,
            'candles1h' => $candles1h,
            'candles4h' => $candles4h
        ]);
        
        // Save RSI data to database
        $this->db->saveRSIData($symbol, $fullData);
        
        // Check each alert type if enabled
        if ($this->isAlertTypeEnabled('rsi')) {
            $this->checkRSIAlerts($symbol, $fullData);
        }
        
        if ($this->isAlertTypeEnabled('price')) {
            $this->checkPriceAlerts($symbol, $fullData);
        }
        
        if ($this->isAlertTypeEnabled('support')) {
            $this->checkSupportResistanceAlerts($symbol, $fullData);
        }
        
        if ($this->isAlertTypeEnabled('second_rejection')) {
            $this->checkSecondResistanceRejection($symbol, $fullData);
        }
        
        if ($this->isAlertTypeEnabled('two_percent_drop')) {
            $this->checkTwoPercentDrop($symbol, $fullData);
        }
        
        // Volume spike detection
        if ($this->settings['directional_volume_spike_enabled']) {
            $this->checkVolumeSpikes($symbol, $candles15m);
        }
        
        // Accumulation/distribution
        if ($this->settings['accumulation_distribution_enabled']) {
            $this->checkVolumeProfile($symbol, $candles1h);
        }
        
        // Pattern recognition
        if ($this->settings['pattern_recognition_enabled']) {
            $this->checkPatterns($symbol, $candles15m, $candles1h, $candles4h, $rsiData);
        }
        
        // Tracked coin specific alerts
        if (in_array($symbol, $trackedSymbols)) {
            $this->checkTrackedCoinEvents($symbol, $fullData);
        }
    }
    
    /**
     * Calculate RSI
     */
    private function calculateRSI($symbol, $candles15m, $candles1d) {
        $rsi15m = null;
        $rsi1d = null;
        
        if ($candles15m && count($candles15m) > 7) {
            $closes15m = array_column($candles15m, 4); // Close price is index 4
            $rsi15m = calculateWilderRSI($closes15m, 7);
        }
        
        if ($candles1d && count($candles1d) > 7) {
            $closes1d = array_column($candles1d, 4);
            $rsi1d = calculateWilderRSI($closes1d, 7);
        }
        
        return [
            'rsi15m' => $rsi15m,
            'rsi1d' => $rsi1d
        ];
    }
    
    /**
     * Calculate support/resistance levels
     */
    private function calculateSupportResistance($candles) {
        if (!$candles || count($candles) < 20) {
            return ['support' => null, 'resistance' => null];
        }
        
        $highs = array_column($candles, 2); // High price is index 2
        $lows = array_column($candles, 3);  // Low price is index 3
        $closes = array_column($candles, 4); // Close price is index 4
        
        $recentHigh = max(array_slice($highs, -10));
        $recentLow = min(array_slice($lows, -10));
        $currentPrice = end($closes);
        
        $pivot = ($recentHigh + $recentLow + $currentPrice) / 3;
        $resistance1 = (2 * $pivot) - $recentLow;
        $support1 = (2 * $pivot) - $recentHigh;
        
        return [
            'support' => min($support1, $pivot - ($recentHigh - $recentLow) * 0.5),
            'resistance' => max($resistance1, $pivot + ($recentHigh - $recentLow) * 0.5)
        ];
    }
    
    /**
     * Analyze trend
     */
    private function analyzeTrend($candles1h) {
        if (!$candles1h || count($candles1h) < 24) {
            return 'neutral';
        }
        
        $closes = array_column($candles1h, 4);
        $ma7 = calculateSMA($closes, 7);
        $ma14 = calculateSMA($closes, 14);
        
        if (!$ma7 || !$ma14) {
            return 'neutral';
        }
        
        $currentPrice = end($closes);
        
        if ($currentPrice > $ma7 && $ma7 > $ma14) {
            return 'bullish';
        } elseif ($currentPrice < $ma7 && $ma7 < $ma14) {
            return 'bearish';
        }
        
        return 'neutral';
    }
    
    /**
     * Check RSI alerts
     */
    private function checkRSIAlerts($symbol, $data) {
        $threshold = $this->settings['rsi_threshold'] ?? 90;
        
        if ($this->settings['rsi7_15m_enabled'] && isset($data['rsi15m']) && $data['rsi15m'] > $threshold) {
            $this->sendAlert(
                $symbol,
                'rsi',
                'RSI 7 (15m) Alert',
                "RSI {$data['rsi15m']} > {$threshold}"
            );
        }
        
        if ($this->settings['rsi7_1d_enabled'] && isset($data['rsi1d']) && $data['rsi1d'] > $threshold) {
            $this->sendAlert(
                $symbol,
                'rsi',
                'RSI 7 (1D) Alert',
                "RSI {$data['rsi1d']} > {$threshold}"
            );
        }
    }
    
    /**
     * Check price alerts
     */
    private function checkPriceAlerts($symbol, $data) {
        if (!isset($data['price'], $data['high24h'])) {
            return;
        }
        
        $breakThreshold = $this->settings['price_break_threshold'] ?? 0.4;
        $breakPercentage = (($data['price'] - $data['high24h']) / $data['high24h']) * 100;
        
        if ($breakPercentage > $breakThreshold) {
            $this->sendAlert(
                $symbol,
                'price',
                '24H High Break',
                "Price broke 24h high by {$breakPercentage}% to $" . formatPrice($data['price'])
            );
        }
    }
    
    /**
     * Check support/resistance alerts
     */
    private function checkSupportResistanceAlerts($symbol, $data) {
        if (!isset($data['price'], $data['support'], $data['resistance'])) {
            return;
        }
        
        $threshold = $this->settings['support_resistance_threshold'] ?? 0.4;
        
        $supportDistance = abs(($data['price'] - $data['support']) / $data['support']) * 100;
        if ($supportDistance <= $threshold) {
            $this->sendAlert(
                $symbol,
                'support',
                'Support Level',
                "Price near support at $" . formatPrice($data['support']) . " ({$supportDistance}%)"
            );
        }
        
        $resistanceDistance = abs(($data['price'] - $data['resistance']) / $data['resistance']) * 100;
        if ($resistanceDistance <= $threshold) {
            $this->sendAlert(
                $symbol,
                'support',
                'Resistance Level',
                "Price near resistance at $" . formatPrice($data['resistance']) . " ({$resistanceDistance}%)"
            );
        }
    }
    
    /**
     * Check second resistance rejection
     */
    private function checkSecondResistanceRejection($symbol, $data) {
        // This would need to track history - simplified version
        if (!isset($data['price'], $data['resistance'])) {
            return;
        }
        
        $threshold = $this->settings['resistance_rejection_threshold'] ?? 0.5;
        $distance = abs(($data['price'] - $data['resistance']) / $data['resistance']) * 100;
        
        if ($distance <= $threshold) {
            // In a real implementation, you'd track rejection counts
            // For now, just send a simple alert
            $this->sendAlert(
                $symbol,
                'second_rejection',
                'Resistance Rejection',
                "Price rejected at resistance $" . formatPrice($data['resistance']) . " ({$distance}% away)"
            );
        }
    }
    
    /**
     * Check 2% drop from 24h high
     */
    private function checkTwoPercentDrop($symbol, $data) {
        if (!isset($data['price'], $data['high24h'])) {
            return;
        }
        
        $dropPercentage = (($data['high24h'] - $data['price']) / $data['high24h']) * 100;
        
        if ($dropPercentage >= 2) {
            $this->sendAlert(
                $symbol,
                'two_percent_drop',
                '📉 2% DROP FROM 24H HIGH',
                "Price dropped {$dropPercentage}% from 24h high\nHigh: $" . formatPrice($data['high24h']) . "\nCurrent: $" . formatPrice($data['price'])
            );
        }
    }
    
    /**
     * Check volume spikes
     */
    private function checkVolumeSpikes($symbol, $candles) {
        if (!$candles || count($candles) < 20) {
            return;
        }
        
        $volumes = array_column($candles, 5); // Volume is index 5
        $avgVolume = array_sum(array_slice($volumes, -20, -1)) / 19;
        
        $candleCount = $this->settings['volume_spike_candle_count'] ?? 3;
        $recentCandles = array_slice($candles, -$candleCount);
        
        foreach ($recentCandles as $candle) {
            $close = floatval($candle[4]);
            $open = floatval($candle[1]);
            $volume = floatval($candle[5]);
            $volumeRatio = $volume / $avgVolume;
            $priceMove = abs(($close - $open) / $open * 100);
            $isGreen = $close > $open;
            
            $threshold = $isGreen ? 
                ($this->settings['buy_volume_spike_threshold'] ?? 3) : 
                ($this->settings['sell_volume_spike_threshold'] ?? 3);
            
            if ($volumeRatio >= $threshold && $priceMove >= ($this->settings['volume_spike_min_price_move'] ?? 1.5)) {
                $type = $isGreen ? 'BUYING_SPIKE' : 'SELLING_SPIKE';
                $spike = [
                    'type' => $type,
                    'volume' => $volume,
                    'avgVolume' => $avgVolume,
                    'volumeRatio' => $volumeRatio,
                    'priceMove' => $priceMove,
                    'price' => $close,
                    'confidence' => min(100, round($volumeRatio * 20))
                ];
                
                $this->db->saveVolumeSpike($symbol, $spike);
                
                $title = $isGreen ? '🚀 BUYING PRESSURE SPIKE' : '📉 SELLING PRESSURE SPIKE';
                $message = "{$volumeRatio}x average volume\nPrice move: {$priceMove}%\nConfidence: {$spike['confidence']}%";
                
                $this->sendAlert($symbol, $type === 'BUYING_SPIKE' ? 'buying_spike' : 'selling_spike', $title, $message);
            }
        }
    }
    
    /**
     * Check volume profile (accumulation/distribution)
     */
    private function checkVolumeProfile($symbol, $candles) {
        if (!$candles || count($candles) < 50) {
            return;
        }
        
        $recentCandles = array_slice($candles, -50);
        $buyingVolume = 0;
        $sellingVolume = 0;
        $neutralVolume = 0;
        $totalVolume = 0;
        
        foreach ($recentCandles as $candle) {
            $volume = floatval($candle[5]);
            $totalVolume += $volume;
            
            if (floatval($candle[4]) > floatval($candle[1])) {
                $buyingVolume += $volume;
            } elseif (floatval($candle[4]) < floatval($candle[1])) {
                $sellingVolume += $volume;
            } else {
                $neutralVolume += $volume;
            }
        }
        
        $buyRatio = $buyingVolume / $totalVolume * 100;
        $sellRatio = $sellingVolume / $totalVolume * 100;
        
        $phase = 'neutral';
        $confidence = 50;
        
        if ($buyRatio > 55) {
            $phase = 'accumulation';
            $confidence = min(100, round(($buyRatio - 50) * 5));
        } elseif ($sellRatio > 55) {
            $phase = 'distribution';
            $confidence = min(100, round(($sellRatio - 50) * 5));
        }
        
        $profile = [
            'phase' => $phase,
            'confidence' => $confidence,
            'buyRatio' => $buyRatio,
            'sellRatio' => $sellRatio,
            'neutralRatio' => $neutralVolume / $totalVolume * 100,
            'totalVolume' => $totalVolume,
            'currentPrice' => floatval(end($recentCandles)[4])
        ];
        
        $this->db->saveVolumeProfile($symbol, $profile);
        
        if ($phase !== 'neutral' && $confidence >= 70) {
            $this->sendAlert(
                $symbol,
                $phase === 'accumulation' ? 'accumulation' : 'distribution',
                $phase === 'accumulation' ? '💰 Accumulation Detected' : '📤 Distribution Detected',
                "{$phase} phase detected\nBuy/Sell: {$buyRatio}% / {$sellRatio}%\nConfidence: {$confidence}%"
            );
        }
    }
    
    /**
     * Check patterns
     */
    private function checkPatterns($symbol, $candles15m, $candles1h, $candles4h, $rsiData) {
        $patterns = [];
        
        // Simple candlestick pattern detection
        if ($this->settings['enable_candlestick_patterns']) {
            $patterns = array_merge($patterns, $this->detectCandlestickPatterns($candles15m));
        }
        
        foreach ($patterns as $pattern) {
            if ($pattern['confidence'] < ($this->settings['min_pattern_confidence'] ?? 60)) {
                continue;
            }
            
            $this->db->savePattern($symbol, $pattern);
            
            $this->sendAlert(
                $symbol,
                'pattern',
                "📊 Pattern Detected: {$pattern['type']}",
                "{$pattern['direction']} pattern\nConfidence: {$pattern['confidence']}%\n{$pattern['description']}"
            );
        }
    }
    
    /**
     * Simple candlestick pattern detection
     */
    private function detectCandlestickPatterns($candles) {
        $patterns = [];
        
        if (!$candles || count($candles) < 2) {
            return $patterns;
        }
        
        $last = end($candles);
        $prev = prev($candles);
        
        $lastOpen = floatval($last[1]);
        $lastClose = floatval($last[4]);
        $lastHigh = floatval($last[2]);
        $lastLow = floatval($last[3]);
        $lastBody = abs($lastClose - $lastOpen);
        $lastRange = $lastHigh - $lastLow;
        $lastIsBullish = $lastClose > $lastOpen;
        
        $prevClose = floatval($prev[4]);
        $prevOpen = floatval($prev[1]);
        $prevIsBullish = $prevClose > $prevOpen;
        
        // Doji
        if ($lastBody < $lastRange * 0.1) {
            $patterns[] = [
                'type' => 'DOJI',
                'direction' => 'neutral',
                'confidence' => 80,
                'description' => 'Indecision, potential reversal',
                'timeframe' => '15m'
            ];
        }
        
        // Engulfing patterns
        if ($lastIsBullish && !$prevIsBullish && 
            $lastClose > $prevHigh && $lastOpen < $prevLow) {
            $patterns[] = [
                'type' => 'BULLISH_ENGULFING',
                'direction' => 'bullish',
                'confidence' => 80,
                'description' => 'Strong bullish reversal signal',
                'timeframe' => '15m'
            ];
        }
        
        if (!$lastIsBullish && $prevIsBullish && 
            $lastOpen > $prevHigh && $lastClose < $prevLow) {
            $patterns[] = [
                'type' => 'BEARISH_ENGULFING',
                'direction' => 'bearish',
                'confidence' => 80,
                'description' => 'Strong bearish reversal signal',
                'timeframe' => '15m'
            ];
        }
        
        return $patterns;
    }
    
    /**
     * Check tracked coin events
     */
    private function checkTrackedCoinEvents($symbol, $data) {
        // This would need to track previous state
        // Simplified version
        if (!isset($data['price'])) {
            return;
        }
        
        // Check support/resistance events
        if (isset($data['support']) && $data['price'] < $data['support'] * 0.99) {
            $this->sendAlert(
                $symbol,
                'support_break',
                '📉 Support Break',
                "Price broke support at $" . formatPrice($data['support'])
            );
        }
        
        if (isset($data['resistance']) && $data['price'] > $data['resistance'] * 1.01) {
            $this->sendAlert(
                $symbol,
                'resistance_break',
                '🚀 Resistance Break',
                "Price broke resistance at $" . formatPrice($data['resistance'])
            );
        }
    }
    
    /**
     * Check order book alerts
     */
    public function checkOrderBookAlerts($userId = null) {
        if ($userId) {
            $this->userId = $userId;
            $this->settings = $this->db->getUserSettings($userId);
        }
        
        if (!$this->settings || !$this->settings['order_book_alerts_enabled']) {
            return;
        }
        
        $symbols = $this->getSymbolsWithVolume();
        $trackedCoins = $this->db->getTrackedCoins($this->userId);
        $trackedSymbols = array_keys($trackedCoins);
        
        foreach ($symbols as $symbolData) {
            $symbol = $symbolData['symbol'];
            
            if ($this->settings['alert_filtering_enabled'] && 
                !empty($this->settings['selectedCoins']) && 
                !in_array($symbol, $this->settings['selectedCoins'])) {
                continue;
            }
            
            try {
                $orderBook = $this->api->getOrderBook($symbol, $this->settings['order_book_depth'] ?? 50);
                $currentPrice = $this->api->getPrice($symbol);
                
                if (!$orderBook || !$currentPrice) {
                    continue;
                }
                
                $analysis = $this->analyzeOrderBook($orderBook, $currentPrice, $symbolData['volume']);
                $this->db->saveOrderBookAnalysis($analysis);
                
                // Check imbalance alerts
                $bearishThreshold = $this->settings['order_book_bearish_threshold'] ?? -15;
                $bullishThreshold = $this->settings['order_book_bullish_threshold'] ?? 15;
                
                if ($analysis['pressure'] === 'bearish' && $analysis['imbalance'] < $bearishThreshold) {
                    $this->sendAlert(
                        $symbol,
                        'orderbook',
                        '⚠️ Bearish Order Book',
                        "Bearish pressure: {$analysis['imbalance']}% imbalance"
                    );
                }
                
                if ($analysis['pressure'] === 'bullish' && $analysis['imbalance'] > $bullishThreshold) {
                    $this->sendAlert(
                        $symbol,
                        'orderbook',
                        '🚀 Bullish Order Book',
                        "Bullish pressure: +{$analysis['imbalance']}% imbalance"
                    );
                }
                
                // Check whale orders
                if ($this->isAlertTypeEnabled('whale')) {
                    $this->checkWhaleOrders($symbol, $orderBook, $currentPrice);
                }
                
            } catch (Exception $e) {
                // Log error and continue
                error_log("Order book error for {$symbol}: " . $e->getMessage());
            }
        }
    }
    
    /**
     * Analyze order book
     */
    private function analyzeOrderBook($orderBook, $currentPrice, $volume) {
        $bids = array_slice($orderBook['bids'], 0, 10);
        $asks = array_slice($orderBook['asks'], 0, 10);
        
        $totalBidVolume = 0;
        $totalAskVolume = 0;
        
        foreach ($bids as $bid) {
            $totalBidVolume += floatval($bid[1]);
        }
        
        foreach ($asks as $ask) {
            $totalAskVolume += floatval($ask[1]);
        }
        
        $totalVolume = $totalBidVolume + $totalAskVolume;
        $imbalance = $totalVolume > 0 ? (($totalBidVolume - $totalAskVolume) / $totalVolume) * 100 : 0;
        
        $pressure = 'neutral';
        if ($imbalance > 10) $pressure = 'bullish';
        elseif ($imbalance < -10) $pressure = 'bearish';
        
        $spread = 0;
        if (!empty($asks) && !empty($bids)) {
            $bestAsk = floatval($asks[0][0]);
            $bestBid = floatval($bids[0][0]);
            $spread = (($bestAsk - $bestBid) / $bestBid) * 100;
        }
        
        return [
            'symbol' => $orderBook['symbol'],
            'current_price' => $currentPrice,
            'volume' => $volume,
            'bid_volume' => $totalBidVolume,
            'ask_volume' => $totalAskVolume,
            'imbalance' => $imbalance,
            'pressure' => $pressure,
            'spread' => $spread,
            'bids' => $bids,
            'asks' => $asks
        ];
    }
    
    /**
     * Check whale orders
     */
    private function checkWhaleOrders($symbol, $orderBook, $currentPrice) {
        $threshold = $this->settings['whale_order_threshold'] ?? 1000000;
        
        $largeBids = [];
        $largeAsks = [];
        $totalBidValue = 0;
        $totalAskValue = 0;
        
        foreach ($orderBook['bids'] as $bid) {
            $value = floatval($bid[0]) * floatval($bid[1]);
            if ($value >= $threshold) {
                $largeBids[] = [
                    'price' => floatval($bid[0]),
                    'quantity' => floatval($bid[1]),
                    'value' => $value,
                    'distance' => (($currentPrice - floatval($bid[0])) / $currentPrice * 100)
                ];
                $totalBidValue += $value;
            }
        }
        
        foreach ($orderBook['asks'] as $ask) {
            $value = floatval($ask[0]) * floatval($ask[1]);
            if ($value >= $threshold) {
                $largeAsks[] = [
                    'price' => floatval($ask[0]),
                    'quantity' => floatval($ask[1]),
                    'value' => $value,
                    'distance' => ((floatval($ask[0]) - $currentPrice) / $currentPrice * 100)
                ];
                $totalAskValue += $value;
            }
        }
        
        if (!empty($largeBids) || !empty($largeAsks)) {
            $whaleData = [
                'symbol' => $symbol,
                'current_price' => $currentPrice,
                'bids' => array_slice($largeBids, 0, 3),
                'asks' => array_slice($largeAsks, 0, 3),
                'total_bid_value' => $totalBidValue,
                'total_ask_value' => $totalAskValue,
                'bid_count' => count($largeBids),
                'ask_count' => count($largeAsks)
            ];
            
            $this->db->saveWhaleOrders($whaleData);
            
            if (!empty($largeBids) && !empty($largeAsks)) {
                $this->sendAlert(
                    $symbol,
                    'whale',
                    '🐋 WHALE BATTLE',
                    "BID: " . formatNumber($totalBidValue) . " | ASK: " . formatNumber($totalAskValue)
                );
            } elseif (!empty($largeBids)) {
                $this->sendAlert(
                    $symbol,
                    'whale',
                    '🐋 WHALE ACCUMULATION',
                    count($largeBids) . " whale BUY orders\nLargest: $" . formatNumber($largeBids[0]['value'])
                );
            } elseif (!empty($largeAsks)) {
                $this->sendAlert(
                    $symbol,
                    'whale',
                    '🐋 WHALE DISTRIBUTION',
                    count($largeAsks) . " whale SELL orders\nLargest: $" . formatNumber($largeAsks[0]['value'])
                );
            }
        }
    }
    
    /**
     * Check liquidation alerts
     */
    public function checkLiquidationAlerts($userId = null) {
        if ($userId) {
            $this->userId = $userId;
            $this->settings = $this->db->getUserSettings($userId);
        }
        
        if (!$this->settings || !$this->isAlertTypeEnabled('liquidation')) {
            return;
        }
        
        try {
            $fiveMinutesAgo = (time() - 300) * 1000; // Convert to milliseconds
            $liquidations = $this->api->getForceOrders($fiveMinutesAgo, 200);
            
            $liquidationMap = [];
            
            foreach ($liquidations as $item) {
                $symbol = $item['symbol'];
                $value = floatval($item['executedQty']) * floatval($item['averagePrice']);
                
                if (!isset($liquidationMap[$symbol])) {
                    $liquidationMap[$symbol] = [
                        'symbol' => $symbol,
                        'count' => 0,
                        'total_value' => 0,
                        'buy_count' => 0,
                        'sell_count' => 0,
                        'max_value' => 0,
                        'average_price' => floatval($item['averagePrice'])
                    ];
                }
                
                $liquidationMap[$symbol]['count']++;
                $liquidationMap[$symbol]['total_value'] += $value;
                $liquidationMap[$symbol]['max_value'] = max($liquidationMap[$symbol]['max_value'], $value);
                
                if ($item['side'] === 'BUY') {
                    $liquidationMap[$symbol]['buy_count']++;
                } else {
                    $liquidationMap[$symbol]['sell_count']++;
                }
            }
            
            foreach ($liquidationMap as $symbol => $data) {
                try {
                    $currentPrice = $this->api->getPrice($symbol);
                    if ($currentPrice) {
                        $data['current_price'] = $currentPrice;
                    }
                } catch (Exception $e) {
                    // Ignore price fetch errors
                }
                
                $this->db->saveLiquidationData($data);
                
                if ($data['total_value'] >= ($this->settings['liquidation_alert_threshold'] ?? 1000000)) {
                    $this->sendAlert(
                        $symbol,
                        'liquidation',
                        '⚠️ Large Liquidation',
                        "Total: $" . formatNumber($data['total_value']) . "\nCount: {$data['count']} positions"
                    );
                }
            }
            
        } catch (Exception $e) {
            error_log("Liquidation check error: " . $e->getMessage());
        }
    }
    
    /**
     * Get symbols with volume
     */
    private function getSymbolsWithVolume() {
        try {
            $tickers = $this->api->get24hrTicker();
            $minVolume = $this->settings['min_volume'] ?? 1000000;
            
            $symbols = [];
            foreach ($tickers as $ticker) {
                $volume = floatval($ticker['quoteVolume']);
                if ($ticker['symbol'] && $volume >= $minVolume) {
                    $symbols[] = [
                        'symbol' => $ticker['symbol'],
                        'price' => floatval($ticker['lastPrice']),
                        'high24h' => floatval($ticker['highPrice']),
                        'low24h' => floatval($ticker['lowPrice']),
                        'volume' => $volume,
                        'change' => floatval($ticker['priceChangePercent'])
                    ];
                }
            }
            
            // Sort by volume descending
            usort($symbols, function($a, $b) {
                return $b['volume'] <=> $a['volume'];
            });
            
            return $symbols;
            
        } catch (Exception $e) {
            error_log("Failed to get symbols with volume: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Check if alert type is enabled
     */
    private function isAlertTypeEnabled($type) {
        return isset($this->settings['alertTypes'][$type]) && 
               $this->settings['alertTypes'][$type] === true;
    }
    
    /**
     * Send alert
     */
    private function sendAlert($symbol, $type, $title, $message) {
        // Check cooldown (simplified - would need Redis or database for production)
        $cooldownKey = "cooldown_{$this->userId}_{$symbol}_{$type}";
        $lastAlert = isset($_SESSION[$cooldownKey]) ? $_SESSION[$cooldownKey] : 0;
        $now = time();
        
        if ($now - $lastAlert < ($this->settings['notification_cooldown'] ?? 30)) {
            return;
        }
        
        $_SESSION[$cooldownKey] = $now;
        
        // Save to database
        $alertData = [
            'title' => $title,
            'message' => $message
        ];
        
        $alertId = $this->db->addAlert($this->userId, $symbol, $type, $title, $message, $alertData);
        
        if ($alertId) {
            // Send email if enabled
            if ($this->settings['send_emails']) {
                $this->notifier->sendEmailAlert($this->settings['email'], $symbol, $type, $title, $message);
            }
            
            // Browser notification (for web app)
            $this->notifier->addBrowserNotification($this->userId, [
                'id' => $alertId,
                'symbol' => $symbol,
                'type' => $type,
                'title' => $title,
                'message' => $message,
                'timestamp' => $now
            ]);
        }
    }
}