<?php
// includes/Database.php

class Database {
    private static $instance = null;
    private $pdo;
    
    private function __construct() {
        global $pdo;
        $this->pdo = $pdo;
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->pdo;
    }
    
    /**
     * Get user settings
     */
    public function getUserSettings($userId) {
        $stmt = $this->pdo->prepare("
            SELECT s.*, u.email, u.username 
            FROM user_settings s
            JOIN users u ON u.id = s.user_id
            WHERE s.user_id = ?
        ");
        $stmt->execute([$userId]);
        $settings = $stmt->fetch();
        
        if (!$settings) {
            // Create default settings if not exists
            $this->createDefaultSettings($userId);
            return $this->getUserSettings($userId);
        }
        
        // Get alert types
        $stmt = $this->pdo->prepare("
            SELECT alert_type, enabled FROM alert_types_enabled WHERE user_id = ?
        ");
        $stmt->execute([$userId]);
        $alertTypes = [];
        while ($row = $stmt->fetch()) {
            $alertTypes[$row['alert_type']] = (bool)$row['enabled'];
        }
        
        $settings['alertTypes'] = $alertTypes;
        
        return $settings;
    }
    
    /**
     * Create default settings for user
     */
    private function createDefaultSettings($userId) {
        $stmt = $this->pdo->prepare("
            INSERT INTO user_settings (user_id) VALUES (?)
        ");
        $stmt->execute([$userId]);
        
        // Insert default alert types
        $defaultTypes = [
            'rsi', 'price', 'support', 'orderbook', 'liquidation',
            'multi', 'volume', 'funding', 'oi_change', 'whale',
            'breakout', 'second_rejection', 'two_percent_drop',
            'buying_spike', 'selling_spike', 'accumulation',
            'distribution', 'volume_climax', 'pattern'
        ];
        
        $stmt = $this->pdo->prepare("
            INSERT INTO alert_types_enabled (user_id, alert_type, enabled) VALUES (?, ?, 1)
        ");
        
        foreach ($defaultTypes as $type) {
            $stmt->execute([$userId, $type]);
        }
    }
    
    /**
     * Update user settings
     */
    public function updateUserSettings($userId, $settings) {
        $fields = [];
        $params = [];
        
        foreach ($settings as $key => $value) {
            if ($key !== 'alertTypes' && $key !== 'email' && $key !== 'username') {
                $fields[] = "$key = ?";
                $params[] = $value;
            }
        }
        
        if (!empty($fields)) {
            $params[] = $userId;
            $sql = "UPDATE user_settings SET " . implode(', ', $fields) . " WHERE user_id = ?";
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
        }
        
        // Update alert types
        if (isset($settings['alertTypes'])) {
            $stmt = $this->pdo->prepare("
                UPDATE alert_types_enabled SET enabled = ? WHERE user_id = ? AND alert_type = ?
            ");
            
            foreach ($settings['alertTypes'] as $type => $enabled) {
                $stmt->execute([$enabled ? 1 : 0, $userId, $type]);
            }
        }
    }
    
    /**
     * Get tracked coins for user
     */
    public function getTrackedCoins($userId) {
        $stmt = $this->pdo->prepare("
            SELECT tc.*, 
                   GROUP_CONCAT(DISTINCT CONCAT(srl.level_type, ':', srl.level_price, ':', srl.strength) SEPARATOR '|') as levels
            FROM tracked_coins tc
            LEFT JOIN support_resistance_levels srl ON srl.tracked_coin_id = tc.id
            WHERE tc.user_id = ?
            GROUP BY tc.id
        ");
        $stmt->execute([$userId]);
        
        $coins = [];
        while ($row = $stmt->fetch()) {
            $coin = $row;
            $coin['supportLevels'] = [];
            $coin['resistanceLevels'] = [];
            
            if ($row['levels']) {
                $levels = explode('|', $row['levels']);
                foreach ($levels as $level) {
                    list($type, $price, $strength) = explode(':', $level);
                    $levelData = [
                        'level' => floatval($price),
                        'strength' => intval($strength)
                    ];
                    if ($type === 'support') {
                        $coin['supportLevels'][] = $levelData;
                    } else {
                        $coin['resistanceLevels'][] = $levelData;
                    }
                }
            }
            
            $coins[$row['symbol']] = $coin;
        }
        
        return $coins;
    }
    
    /**
     * Add tracked coin
     */
    public function addTrackedCoin($userId, $symbol, $data) {
        $stmt = $this->pdo->prepare("
            INSERT INTO tracked_coins 
            (user_id, symbol, current_price, volume_24h, trend, price_action, volatility, volume_ratio, last_update)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ON DUPLICATE KEY UPDATE
            current_price = VALUES(current_price),
            volume_24h = VALUES(volume_24h),
            trend = VALUES(trend),
            price_action = VALUES(price_action),
            volatility = VALUES(volatility),
            volume_ratio = VALUES(volume_ratio),
            last_update = NOW()
        ");
        
        $stmt->execute([
            $userId, $symbol, $data['current_price'], $data['volume_24h'],
            $data['trend'], $data['price_action'], $data['volatility'],
            $data['volume_ratio']
        ]);
        
        $coinId = $this->pdo->lastInsertId();
        
        // Add support/resistance levels
        if (isset($data['supportLevels'])) {
            $this->addSupportResistanceLevels($coinId, 'support', $data['supportLevels']);
        }
        
        if (isset($data['resistanceLevels'])) {
            $this->addSupportResistanceLevels($coinId, 'resistance', $data['resistanceLevels']);
        }
        
        // Add to history
        $this->addCoinHistory($coinId, $data);
        
        return $coinId;
    }
    
    /**
     * Add support/resistance levels
     */
    private function addSupportResistanceLevels($coinId, $type, $levels) {
        $this->pdo->prepare("DELETE FROM support_resistance_levels WHERE tracked_coin_id = ? AND level_type = ?")
            ->execute([$coinId, $type]);
        
        $stmt = $this->pdo->prepare("
            INSERT INTO support_resistance_levels (tracked_coin_id, level_type, level_price, strength, level_order)
            VALUES (?, ?, ?, ?, ?)
        ");
        
        foreach ($levels as $index => $level) {
            $stmt->execute([
                $coinId, $type, $level['level'], $level['strength'], $index
            ]);
        }
    }
    
    /**
     * Add coin history
     */
    private function addCoinHistory($coinId, $data) {
        $stmt = $this->pdo->prepare("
            INSERT INTO coin_tracking_history (tracked_coin_id, price, volume, trend)
            VALUES (?, ?, ?, ?)
        ");
        $stmt->execute([
            $coinId, $data['current_price'], $data['volume_24h'], $data['trend']
        ]);
    }
    
    /**
     * Remove tracked coin
     */
    public function removeTrackedCoin($userId, $symbol) {
        $stmt = $this->pdo->prepare("
            DELETE FROM tracked_coins WHERE user_id = ? AND symbol = ?
        ");
        return $stmt->execute([$userId, $symbol]);
    }
    
    /**
     * Add alert
     */
    public function addAlert($userId, $symbol, $type, $title, $message, $data = null) {
        $stmt = $this->pdo->prepare("
            INSERT INTO alerts (user_id, symbol, alert_type, title, message, data)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([
            $userId, $symbol, $type, $title, $message,
            $data ? json_encode($data) : null
        ]);
    }
    
    /**
     * Get recent alerts for user
     */
    public function getRecentAlerts($userId, $limit = 100, $symbol = null, $type = null) {
        $sql = "SELECT * FROM alerts WHERE user_id = ?";
        $params = [$userId];
        
        if ($symbol) {
            $sql .= " AND symbol = ?";
            $params[] = $symbol;
        }
        
        if ($type) {
            $sql .= " AND alert_type = ?";
            $params[] = $type;
        }
        
        $sql .= " ORDER BY created_at DESC LIMIT ?";
        $params[] = $limit;
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        
        $alerts = [];
        while ($row = $stmt->fetch()) {
            if ($row['data']) {
                $row['data'] = json_decode($row['data'], true);
            }
            $alerts[] = $row;
        }
        
        return $alerts;
    }
    
    /**
     * Get alerts by coin
     */
    public function getAlertsByCoin($userId, $limit = 100) {
        $stmt = $this->pdo->prepare("
            SELECT * FROM alerts 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT ?
        ");
        $stmt->execute([$userId, $limit * 10]); // Get more for grouping
        
        $alertsByCoin = [];
        while ($row = $stmt->fetch()) {
            if ($row['data']) {
                $row['data'] = json_decode($row['data'], true);
            }
            
            if (!isset($alertsByCoin[$row['symbol']])) {
                $alertsByCoin[$row['symbol']] = [];
            }
            
            $alertsByCoin[$row['symbol']][] = $row;
            
            // Limit per coin
            if (count($alertsByCoin[$row['symbol']]) > $limit) {
                array_pop($alertsByCoin[$row['symbol']]);
            }
        }
        
        return $alertsByCoin;
    }
    
    /**
     * Clear alerts
     */
    public function clearAlerts($userId, $symbol = null) {
        if ($symbol) {
            $stmt = $this->pdo->prepare("DELETE FROM alerts WHERE user_id = ? AND symbol = ?");
            return $stmt->execute([$userId, $symbol]);
        } else {
            $stmt = $this->pdo->prepare("DELETE FROM alerts WHERE user_id = ?");
            return $stmt->execute([$userId]);
        }
    }
    
    /**
     * Save RSI data
     */
    public function saveRSIData($symbol, $data) {
        $stmt = $this->pdo->prepare("
            INSERT INTO rsi_data 
            (symbol, rsi_15m, rsi_1d, price, volume_24h, change_24h, support, resistance, trend)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([
            $symbol,
            $data['rsi15m'] ?? null,
            $data['rsi1d'] ?? null,
            $data['price'] ?? null,
            $data['volume'] ?? null,
            $data['change'] ?? null,
            $data['support'] ?? null,
            $data['resistance'] ?? null,
            $data['trend'] ?? null
        ]);
    }
    
    /**
     * Save order book analysis
     */
    public function saveOrderBookAnalysis($data) {
        $stmt = $this->pdo->prepare("
            INSERT INTO order_book_analysis 
            (symbol, current_price, volume_24h, bid_volume, ask_volume, imbalance, pressure, spread, bids, asks, ai_analysis)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([
            $data['symbol'],
            $data['current_price'] ?? null,
            $data['volume'] ?? null,
            $data['bid_volume'] ?? null,
            $data['ask_volume'] ?? null,
            $data['imbalance'] ?? null,
            $data['pressure'] ?? null,
            $data['spread'] ?? null,
            $data['bids'] ? json_encode($data['bids']) : null,
            $data['asks'] ? json_encode($data['asks']) : null,
            $data['ai_analysis'] ?? null
        ]);
    }
    
    /**
     * Save liquidation data
     */
    public function saveLiquidationData($data) {
        $stmt = $this->pdo->prepare("
            INSERT INTO liquidation_data 
            (symbol, count, total_value, buy_count, sell_count, average_price, current_price, max_value)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([
            $data['symbol'],
            $data['count'] ?? 0,
            $data['total_value'] ?? 0,
            $data['buy_count'] ?? 0,
            $data['sell_count'] ?? 0,
            $data['average_price'] ?? null,
            $data['current_price'] ?? null,
            $data['max_value'] ?? 0
        ]);
    }
    
    /**
     * Save volume spike
     */
    public function saveVolumeSpike($symbol, $spike) {
        $stmt = $this->pdo->prepare("
            INSERT INTO volume_spike_history 
            (symbol, spike_type, volume, avg_volume, volume_ratio, price_move, price, confidence, candle_count, direction, description)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([
            $symbol,
            $spike['type'] ?? null,
            $spike['volume'] ?? null,
            $spike['avgVolume'] ?? null,
            $spike['volumeRatio'] ?? null,
            $spike['priceMove'] ?? null,
            $spike['price'] ?? null,
            $spike['confidence'] ?? null,
            $spike['candleCount'] ?? null,
            $spike['direction'] ?? null,
            $spike['description'] ?? null
        ]);
    }
    
    /**
     * Save volume profile
     */
    public function saveVolumeProfile($symbol, $profile) {
        $stmt = $this->pdo->prepare("
            INSERT INTO volume_profile 
            (symbol, phase, confidence, buy_ratio, sell_ratio, neutral_ratio, total_volume, price_change, vwap, vwap_distance, current_price)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([
            $symbol,
            $profile['phase'] ?? null,
            $profile['confidence'] ?? null,
            $profile['buyRatio'] ?? null,
            $profile['sellRatio'] ?? null,
            $profile['neutralRatio'] ?? null,
            $profile['totalVolume'] ?? null,
            $profile['priceChange'] ?? null,
            $profile['vwap'] ?? null,
            $profile['vwapDistance'] ?? null,
            $profile['currentPrice'] ?? null
        ]);
    }
    
    /**
     * Save detected pattern
     */
    public function savePattern($symbol, $pattern) {
        $stmt = $this->pdo->prepare("
            INSERT INTO detected_patterns 
            (symbol, pattern_type, direction, confidence, timeframe, description, data)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        
        $data = $pattern;
        unset($data['type'], $data['direction'], $data['confidence'], 
              $data['timeframe'], $data['description']);
        
        return $stmt->execute([
            $symbol,
            $pattern['type'] ?? null,
            $pattern['direction'] ?? null,
            $pattern['confidence'] ?? null,
            $pattern['timeframe'] ?? null,
            $pattern['description'] ?? null,
            json_encode($data)
        ]);
    }
    
    /**
     * Save funding rate
     */
    public function saveFundingRate($symbol, $rate) {
        $stmt = $this->pdo->prepare("
            INSERT INTO funding_rates (symbol, rate) VALUES (?, ?)
        ");
        return $stmt->execute([$symbol, $rate]);
    }
    
    /**
     * Save open interest
     */
    public function saveOpenInterest($symbol, $oi) {
        $stmt = $this->pdo->prepare("
            INSERT INTO open_interest (symbol, open_interest) VALUES (?, ?)
        ");
        return $stmt->execute([$symbol, $oi]);
    }
    
    /**
     * Save whale orders
     */
    public function saveWhaleOrders($data) {
        $stmt = $this->pdo->prepare("
            INSERT INTO whale_orders 
            (symbol, current_price, bids, asks, total_bid_value, total_ask_value, bid_count, ask_count)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([
            $data['symbol'],
            $data['current_price'] ?? null,
            $data['bids'] ? json_encode($data['bids']) : null,
            $data['asks'] ? json_encode($data['asks']) : null,
            $data['total_bid_value'] ?? 0,
            $data['total_ask_value'] ?? 0,
            $data['bid_count'] ?? 0,
            $data['ask_count'] ?? 0
        ]);
    }
}