<?php
// includes/NotificationService.php

class NotificationService {
    private $pdo;
    
    public function __construct() {
        $this->pdo = Database::getInstance()->getConnection();
    }
    
    /**
     * Send email alert
     */
    public function sendEmailAlert($to, $symbol, $type, $title, $message) {
        $subject = "Binance Futures Alert: {$symbol} - {$title}";
        
        $htmlMessage = $this->buildEmailHTML($symbol, $type, $title, $message);
        
        return $this->sendEmail($to, $subject, $htmlMessage);
    }
    
    /**
     * Build email HTML
     */
    private function buildEmailHTML($symbol, $type, $title, $message) {
        $typeColors = [
            'rsi' => '#3b82f6',
            'price' => '#10b981',
            'support' => '#f59e0b',
            'orderbook' => '#8b5cf6',
            'liquidation' => '#ef4444',
            'whale' => '#6366f1',
            'buying_spike' => '#10b981',
            'selling_spike' => '#ef4444',
            'accumulation' => '#14b8a6',
            'distribution' => '#f97316'
        ];
        
        $color = $typeColors[$type] ?? '#3b82f6';
        
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <style>
                body { font-family: -apple-system, BlinkMacSystemFont, sans-serif; background: #f5f5f5; padding: 20px; }
                .container { max-width: 600px; margin: 0 auto; background: white; border-radius: 8px; overflow: hidden; }
                .header { background: {$color}; padding: 20px; color: white; }
                .header h1 { margin: 0; font-size: 24px; }
                .header .symbol { font-size: 14px; opacity: 0.9; margin-top: 5px; }
                .content { padding: 20px; }
                .message { background: #f8f9fa; padding: 15px; border-radius: 6px; margin: 15px 0; white-space: pre-line; }
                .footer { background: #f8f9fa; padding: 15px; text-align: center; font-size: 12px; color: #666; }
                .button { display: inline-block; padding: 10px 20px; background: {$color}; color: white; text-decoration: none; border-radius: 6px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>{$title}</h1>
                    <div class='symbol'>{$symbol}</div>
                </div>
                <div class='content'>
                    <div class='message'>{$message}</div>
                    <p style='text-align: center;'>
                        <a href='" . $_ENV['APP_URL'] . "/dashboard.php' class='button'>View Dashboard</a>
                    </p>
                </div>
                <div class='footer'>
                    <p>Binance Futures Alert System</p>
                    <p>To change your alert settings, visit the dashboard.</p>
                </div>
            </div>
        </body>
        </html>
        ";
    }
    
    /**
     * Send email
     */
    private function sendEmail($to, $subject, $htmlMessage) {
        $headers = [
            'MIME-Version: 1.0',
            'Content-type: text/html; charset=utf-8',
            'From: ' . ($_ENV['SMTP_FROM_NAME'] ?? 'Binance Futures Alerts') . ' <' . ($_ENV['SMTP_FROM'] ?? 'noreply@localhost') . '>',
            'X-Mailer: PHP/' . phpversion()
        ];
        
        return mail($to, $subject, $htmlMessage, implode("\r\n", $headers));
    }
    
    /**
     * Add browser notification (stored for web app)
     */
    public function addBrowserNotification($userId, $notification) {
        // Store in session or database for web app to display
        session_start();
        if (!isset($_SESSION['notifications'])) {
            $_SESSION['notifications'] = [];
        }
        
        $_SESSION['notifications'][$userId][] = $notification;
        
        // Keep only last 50
        if (count($_SESSION['notifications'][$userId]) > 50) {
            array_shift($_SESSION['notifications'][$userId]);
        }
    }
    
    /**
     * Get browser notifications for user
     */
    public function getBrowserNotifications($userId) {
        session_start();
        $notifications = $_SESSION['notifications'][$userId] ?? [];
        
        // Clear after retrieving
        unset($_SESSION['notifications'][$userId]);
        
        return $notifications;
    }
}