<?php
// includes/functions.php

/**
 * Format number with K/M/B suffix
 */
function formatNumber($num) {
    if ($num >= 1000000000) {
        return number_format($num / 1000000000, 2) . 'B';
    }
    if ($num >= 1000000) {
        return number_format($num / 1000000, 2) . 'M';
    }
    if ($num >= 1000) {
        return number_format($num / 1000, 2) . 'K';
    }
    return number_format($num, 2);
}

/**
 * Format volume
 */
function formatVolume($volume) {
    if ($volume >= 1000000000) {
        return number_format($volume / 1000000000, 2) . 'B';
    }
    if ($volume >= 1000000) {
        return number_format($volume / 1000000, 2) . 'M';
    }
    if ($volume >= 1000) {
        return number_format($volume / 1000, 2) . 'K';
    }
    return number_format($volume, 2);
}

/**
 * Format price
 */
function formatPrice($price) {
    if ($price === null || $price === 0) {
        return 'N/A';
    }
    
    if ($price >= 1000) {
        return number_format($price, 2);
    } elseif ($price >= 1) {
        return number_format($price, 4);
    } elseif ($price >= 0.01) {
        return number_format($price, 6);
    } elseif ($price >= 0.0001) {
        return number_format($price, 8);
    } else {
        return rtrim(sprintf('%.12f', $price), '0');
    }
}

/**
 * Format time ago
 */
function formatTimeAgo($timestamp) {
    $now = time();
    $diff = $now - (is_numeric($timestamp) ? $timestamp : strtotime($timestamp));
    
    if ($diff < 60) {
        return 'just now';
    }
    if ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' minute' . ($mins > 1 ? 's' : '') . ' ago';
    }
    if ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
    }
    if ($diff < 2592000) {
        $days = floor($diff / 86400);
        return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
    }
    
    return date('Y-m-d H:i', $timestamp);
}

/**
 * Format time short
 */
function formatTimeShort($timestamp) {
    $now = time();
    $diff = $now - (is_numeric($timestamp) ? $timestamp : strtotime($timestamp));
    
    if ($diff < 60) {
        return '<1m';
    }
    if ($diff < 3600) {
        return floor($diff / 60) . 'm';
    }
    if ($diff < 86400) {
        return floor($diff / 3600) . 'h';
    }
    return date('H:i', $timestamp);
}

/**
 * Calculate SMA
 */
function calculateSMA($prices, $period) {
    if (count($prices) < $period) {
        return null;
    }
    $recent = array_slice($prices, -$period);
    return array_sum($recent) / $period;
}

/**
 * Calculate Wilder's RSI
 */
function calculateWilderRSI($closes, $period = 7) {
    if (count($closes) < $period + 1) {
        return null;
    }
    
    $gains = 0;
    $losses = 0;
    
    for ($i = 1; $i <= $period; $i++) {
        $diff = $closes[$i] - $closes[$i - 1];
        if ($diff >= 0) {
            $gains += $diff;
        } else {
            $losses -= $diff;
        }
    }
    
    $avgGain = $gains / $period;
    $avgLoss = $losses / $period;
    
    for ($i = $period + 1; $i < count($closes); $i++) {
        $diff = $closes[$i] - $closes[$i - 1];
        $gain = $diff > 0 ? $diff : 0;
        $loss = $diff < 0 ? -$diff : 0;
        
        $avgGain = ($avgGain * ($period - 1) + $gain) / $period;
        $avgLoss = ($avgLoss * ($period - 1) + $loss) / $period;
    }
    
    if ($avgLoss == 0) {
        return 100;
    }
    
    $rs = $avgGain / $avgLoss;
    return round(100 - (100 / (1 + $rs)), 2);
}

/**
 * Calculate linear regression
 */
function calculateLinearRegression($xValues, $yValues) {
    $n = count($xValues);
    $sumX = array_sum($xValues);
    $sumY = array_sum($yValues);
    
    $sumXY = 0;
    $sumX2 = 0;
    
    for ($i = 0; $i < $n; $i++) {
        $sumXY += $xValues[$i] * $yValues[$i];
        $sumX2 += $xValues[$i] * $xValues[$i];
    }
    
    $slope = ($n * $sumXY - $sumX * $sumY) / ($n * $sumX2 - $sumX * $sumX);
    $intercept = ($sumY - $slope * $sumX) / $n;
    
    return ['slope' => $slope, 'intercept' => $intercept];
}

/**
 * Check if user is logged in
 */
function isLoggedIn() {
    session_start();
    return isset($_SESSION['user_id']);
}

/**
 * Redirect if not logged in
 */
function requireLogin() {
    if (!isLoggedIn()) {
        header('Location: login.php');
        exit;
    }
}

/**
 * Generate CSRF token
 */
function generateCSRFToken() {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Verify CSRF token
 */
function verifyCSRFToken($token) {
    if (empty($_SESSION['csrf_token']) || $token !== $_SESSION['csrf_token']) {
        die('Invalid CSRF token');
    }
}

/**
 * Hash password
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Verify password
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}