<?php
// install.php - Run this first to set up the database

require_once 'config/database.php';

$sql = "
-- Users table
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- User settings table
CREATE TABLE IF NOT EXISTS user_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNIQUE NOT NULL,
    enabled BOOLEAN DEFAULT TRUE,
    show_notifications BOOLEAN DEFAULT TRUE,
    send_emails BOOLEAN DEFAULT TRUE,
    notification_cooldown INT DEFAULT 30,
    rsi_check_interval INT DEFAULT 70,
    order_book_interval INT DEFAULT 300,
    liquidation_interval INT DEFAULT 120,
    rsi7_15m_enabled BOOLEAN DEFAULT TRUE,
    rsi7_1d_enabled BOOLEAN DEFAULT TRUE,
    rsi_threshold INT DEFAULT 90,
    price_alerts_enabled BOOLEAN DEFAULT TRUE,
    price_break_threshold DECIMAL(5,2) DEFAULT 0.4,
    price_increase_threshold INT DEFAULT 14,
    support_resistance_alerts_enabled BOOLEAN DEFAULT TRUE,
    support_resistance_threshold DECIMAL(5,2) DEFAULT 0.4,
    order_book_alerts_enabled BOOLEAN DEFAULT TRUE,
    order_book_depth INT DEFAULT 50,
    order_book_bearish_threshold INT DEFAULT -15,
    order_book_bullish_threshold INT DEFAULT 15,
    ai_analysis_enabled BOOLEAN DEFAULT TRUE,
    min_volume BIGINT DEFAULT 1000000,
    min_order_book_volume BIGINT DEFAULT 10000000,
    liquidation_alert_threshold BIGINT DEFAULT 1000000,
    multi_condition_alerts_enabled BOOLEAN DEFAULT TRUE,
    volume_spike_threshold DECIMAL(5,2) DEFAULT 5,
    liquidation_cluster_threshold INT DEFAULT 3,
    funding_rate_alert_threshold DECIMAL(10,4) DEFAULT -0.01,
    oi_change_threshold INT DEFAULT 20,
    whale_order_threshold BIGINT DEFAULT 1000000,
    breakout_volume_multiplier DECIMAL(5,2) DEFAULT 2,
    second_rejection_alerts_enabled BOOLEAN DEFAULT TRUE,
    resistance_rejection_threshold DECIMAL(5,2) DEFAULT 0.5,
    two_percent_drop_alerts_enabled BOOLEAN DEFAULT TRUE,
    coin_tracking_enabled BOOLEAN DEFAULT TRUE,
    coin_tracking_interval INT DEFAULT 60,
    pullback_threshold DECIMAL(5,2) DEFAULT 2,
    reversal_threshold DECIMAL(5,2) DEFAULT 3,
    tracked_volume_spike_multiplier DECIMAL(5,2) DEFAULT 3,
    resistance_support_levels INT DEFAULT 3,
    track_coin_history_days INT DEFAULT 7,
    directional_volume_spike_enabled BOOLEAN DEFAULT TRUE,
    buy_volume_spike_threshold DECIMAL(5,2) DEFAULT 3,
    sell_volume_spike_threshold DECIMAL(5,2) DEFAULT 3,
    volume_spike_candle_count INT DEFAULT 3,
    volume_spike_min_price_move DECIMAL(5,2) DEFAULT 1.5,
    accumulation_distribution_enabled BOOLEAN DEFAULT TRUE,
    pattern_recognition_enabled BOOLEAN DEFAULT TRUE,
    pattern_check_interval INT DEFAULT 300,
    min_pattern_confidence INT DEFAULT 60,
    pattern_alert_cooldown INT DEFAULT 3600,
    enable_chart_patterns BOOLEAN DEFAULT TRUE,
    enable_candlestick_patterns BOOLEAN DEFAULT TRUE,
    enable_divergence_patterns BOOLEAN DEFAULT TRUE,
    enable_market_structure_patterns BOOLEAN DEFAULT TRUE,
    enable_volume_profile_patterns BOOLEAN DEFAULT TRUE,
    alert_filtering_enabled BOOLEAN DEFAULT TRUE,
    max_alerts_per_coin INT DEFAULT 50,
    group_alerts_by_coin BOOLEAN DEFAULT TRUE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Alert types enabled table
CREATE TABLE IF NOT EXISTS alert_types_enabled (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    alert_type VARCHAR(50) NOT NULL,
    enabled BOOLEAN DEFAULT TRUE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_alert (user_id, alert_type)
);

-- Tracked coins table
CREATE TABLE IF NOT EXISTS tracked_coins (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    symbol VARCHAR(20) NOT NULL,
    added_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_update TIMESTAMP NULL,
    current_price DECIMAL(20,8),
    volume_24h BIGINT,
    trend VARCHAR(20),
    price_action VARCHAR(20),
    volatility DECIMAL(10,4),
    volume_ratio DECIMAL(10,4),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_symbol (user_id, symbol)
);

-- Coin tracking history
CREATE TABLE IF NOT EXISTS coin_tracking_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tracked_coin_id INT NOT NULL,
    price DECIMAL(20,8),
    volume BIGINT,
    trend VARCHAR(20),
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (tracked_coin_id) REFERENCES tracked_coins(id) ON DELETE CASCADE
);

-- Support/resistance levels for tracked coins
CREATE TABLE IF NOT EXISTS support_resistance_levels (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tracked_coin_id INT NOT NULL,
    level_type ENUM('support', 'resistance') NOT NULL,
    level_price DECIMAL(20,8) NOT NULL,
    strength INT DEFAULT 0,
    level_order INT DEFAULT 0,
    last_tested TIMESTAMP NULL,
    FOREIGN KEY (tracked_coin_id) REFERENCES tracked_coins(id) ON DELETE CASCADE
);

-- Alerts table
CREATE TABLE IF NOT EXISTS alerts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    symbol VARCHAR(20) NOT NULL,
    alert_type VARCHAR(50) NOT NULL,
    title VARCHAR(255),
    message TEXT,
    data JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    email_sent BOOLEAN DEFAULT FALSE,
    browser_notified BOOLEAN DEFAULT FALSE,
    INDEX idx_user_created (user_id, created_at),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Futures symbols cache
CREATE TABLE IF NOT EXISTS futures_symbols (
    id INT AUTO_INCREMENT PRIMARY KEY,
    symbol VARCHAR(20) UNIQUE NOT NULL,
    status VARCHAR(20) DEFAULT 'TRADING',
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- RSI data cache
CREATE TABLE IF NOT EXISTS rsi_data (
    id INT AUTO_INCREMENT PRIMARY KEY,
    symbol VARCHAR(20) NOT NULL,
    rsi_15m DECIMAL(10,4),
    rsi_1d DECIMAL(10,4),
    price DECIMAL(20,8),
    volume_24h BIGINT,
    change_24h DECIMAL(10,4),
    support DECIMAL(20,8),
    resistance DECIMAL(20,8),
    trend VARCHAR(20),
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_symbol_recorded (symbol, recorded_at)
);

-- Order book analysis cache
CREATE TABLE IF NOT EXISTS order_book_analysis (
    id INT AUTO_INCREMENT PRIMARY KEY,
    symbol VARCHAR(20) NOT NULL,
    current_price DECIMAL(20,8),
    volume_24h BIGINT,
    bid_volume DECIMAL(20,8),
    ask_volume DECIMAL(20,8),
    imbalance DECIMAL(10,4),
    pressure VARCHAR(20),
    spread DECIMAL(10,4),
    bids JSON,
    asks JSON,
    ai_analysis TEXT,
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_symbol_recorded (symbol, recorded_at)
);

-- Liquidation data cache
CREATE TABLE IF NOT EXISTS liquidation_data (
    id INT AUTO_INCREMENT PRIMARY KEY,
    symbol VARCHAR(20) NOT NULL,
    count INT DEFAULT 0,
    total_value BIGINT,
    buy_count INT DEFAULT 0,
    sell_count INT DEFAULT 0,
    average_price DECIMAL(20,8),
    current_price DECIMAL(20,8),
    max_value BIGINT,
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_symbol_recorded (symbol, recorded_at)
);

-- Volume spike history
CREATE TABLE IF NOT EXISTS volume_spike_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    symbol VARCHAR(20) NOT NULL,
    spike_type VARCHAR(50) NOT NULL,
    volume BIGINT,
    avg_volume BIGINT,
    volume_ratio DECIMAL(10,4),
    price_move DECIMAL(10,4),
    price DECIMAL(20,8),
    confidence INT,
    candle_count INT,
    direction VARCHAR(20),
    description TEXT,
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_symbol_recorded (symbol, recorded_at)
);

-- Volume profile cache
CREATE TABLE IF NOT EXISTS volume_profile (
    id INT AUTO_INCREMENT PRIMARY KEY,
    symbol VARCHAR(20) NOT NULL,
    phase VARCHAR(20),
    confidence INT,
    buy_ratio DECIMAL(10,4),
    sell_ratio DECIMAL(10,4),
    neutral_ratio DECIMAL(10,4),
    total_volume BIGINT,
    price_change DECIMAL(10,4),
    vwap DECIMAL(20,8),
    vwap_distance DECIMAL(10,4),
    current_price DECIMAL(20,8),
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_symbol_recorded (symbol, recorded_at)
);

-- Detected patterns cache
CREATE TABLE IF NOT EXISTS detected_patterns (
    id INT AUTO_INCREMENT PRIMARY KEY,
    symbol VARCHAR(20) NOT NULL,
    pattern_type VARCHAR(100) NOT NULL,
    direction VARCHAR(20),
    confidence INT,
    timeframe VARCHAR(10),
    description TEXT,
    data JSON,
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_symbol_recorded (symbol, recorded_at)
);

-- Funding rates cache
CREATE TABLE IF NOT EXISTS funding_rates (
    id INT AUTO_INCREMENT PRIMARY KEY,
    symbol VARCHAR(20) NOT NULL,
    rate DECIMAL(20,8),
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_symbol_recorded (symbol, recorded_at)
);

-- Open interest cache
CREATE TABLE IF NOT EXISTS open_interest (
    id INT AUTO_INCREMENT PRIMARY KEY,
    symbol VARCHAR(20) NOT NULL,
    open_interest BIGINT,
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_symbol_recorded (symbol, recorded_at)
);

-- Whale orders cache
CREATE TABLE IF NOT EXISTS whale_orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    symbol VARCHAR(20) NOT NULL,
    current_price DECIMAL(20,8),
    bids JSON,
    asks JSON,
    total_bid_value BIGINT,
    total_ask_value BIGINT,
    bid_count INT,
    ask_count INT,
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_symbol_recorded (symbol, recorded_at)
);

-- Insert default alert types
INSERT IGNORE INTO alert_types_enabled (user_id, alert_type, enabled) VALUES
(1, 'rsi', 1),
(1, 'price', 1),
(1, 'support', 1),
(1, 'orderbook', 1),
(1, 'liquidation', 1),
(1, 'multi', 1),
(1, 'volume', 1),
(1, 'funding', 1),
(1, 'oi_change', 1),
(1, 'whale', 1),
(1, 'breakout', 1),
(1, 'second_rejection', 1),
(1, 'two_percent_drop', 1),
(1, 'buying_spike', 1),
(1, 'selling_spike', 1),
(1, 'accumulation', 1),
(1, 'distribution', 1),
(1, 'volume_climax', 1),
(1, 'pattern', 1);
";

try {
    $pdo->exec($sql);
    echo "Database tables created successfully!\n";
    echo "Please create a .env file with your database credentials.\n";
} catch (PDOException $e) {
    die("Error creating tables: " . $e->getMessage() . "\n");
}